/** @file   datatypes.h
 * @brief   Declares some general datatypes.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/02/08 21:59:43 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_DATATYPES_H
#define H_ENG2D_DATATYPES_H

#include <allegro.h>
#include <string>
#include "eng2d_dll.h"


namespace eng2d {

/** A datatype for blending modes. The drawing methods that support
 * these types use the corresponding blender-functions defined in
 * Allegro library.
 */
enum BlendMode {
  /** Solid drawing. No blending or transparency. */
  ESolid,
  /** Transparent blending */
  ETrans,
  /** Add blending */
  EAdd,
  /** Multiply blending */
  EMultiply,
  /** Dodge blending */
  EDodge,
  /** Burn blending */
  EBurn,
  /** Dissolve blending */
  EDissolve
};

/** Possible types of graphic content. Used particulary when using Allegro's
 * drawing methods with some datastructures defined in this library.
 */
enum GfxContentType {
  /** Normal Allegro bitmap. Should be drawn with blit() function. */
  EBitmap,
  /** Allegro bitmap but should be drawn with draw_sprite() function. */
  ESprite,
  /** Rle encoded sprite. Should be draw with draw_rle_sprite() function. */
  ERleSprite,
  /** Compiled sprite. Not supported yet. */
  ECompiledSprite,
  /** Unspecified. */
  EUnspecified
};


/** Datatype for rgb color component. */
typedef float CVal;



///
/// Error codes
/// ===========

/** No error. */
const int KErrNone = 0;
/** Requested object/target was not found */
const int KErrNotFound = -1;
/** General unspecified error. */
const int KErrGeneral = -2;
/** Unexpected Eof-Of-File encountered. */
const int KErrEof = -3;
/** Error while reading data. */
const int KErrRead = -4;
/** Error while writing data. */
const int KErrWrite = -5;
/** Data has been corrupted. */
const int KErrCorrupt = -6;
/** Requested operation or handled object is not supported. */
const int KErrNotSupported = -7;
/** Unrecoverable IO error has occured. For instance attempt to convert data
 * read from istream to integer has failed.
 */
const int KErrBadIo = -8;
/** If the object in question already exists. */
const int KErrAlreadyExists = -9;


/** An uid that means "no uid" */
const int KInvalidUid = -1;


/** Sets the blending function.
 * @param   aBlendMode          The blending mode
 * @param   aAlpha              Alpha value
 */
inline void DLLIMPORT SetBlendFunction( BlendMode aBlendMode, int aAlpha )
{
  switch ( aBlendMode ) {
    case ( ESolid ): {
      solid_mode();
      break;
    }
    case ( EMultiply ): {
      set_multiply_blender( 0,0,0, aAlpha );
      break;
    }
    case ( EAdd ): {
      set_add_blender( 0,0,0, aAlpha );
      break;
    }
    case ( EDodge ): {
      set_dodge_blender( 0,0,0, aAlpha );
      break;
    }
    case ( EBurn ): {
      set_burn_blender( 0,0,0, aAlpha );
      break;
    }
    case ( ETrans ): {
      set_trans_blender( 0,0,0, aAlpha );
      break;
    }
    case ( EDissolve ): {
      set_dissolve_blender( 0,0,0, aAlpha );
      break;
    }
  }
}


/** Converts the given string to blendmode.
 * @param   aStr                The string containing the blendmode name.
 * @return  A BlendMode that matches the given string. ESolid will be returned
 *          if there is no perfect match.
 */
inline BlendMode String2BlendMode( const std::string& aStr )
{
  if ( aStr == "solid" ) {
    return ESolid;
  } else if ( aStr == "trans" ) {
    return ETrans;
  } else if ( aStr == "add" ) {
    return EAdd;
  } else if ( aStr == "multiply" ) {
    return EMultiply;
  } else if ( aStr == "dodge" ) {
    return EDodge;
  } else if ( aStr == "burn" ) {
    return EBurn;
  } else if ( aStr == "dissolve" ) {
    return EDissolve;
  } else {
    return ESolid;
  }
}



/** Converts the given integer to a string.
 * @param   aInteger            Integer
 * @return  A string representation of the integer.
 */
inline std::string Int2String( int aInteger )
{
  char num[34];
  return std::string( itoa(aInteger, num, 10) );
}


};  // end of namespace

#endif
